<?php
require_once __DIR__.'/../helpers.php';
require_once __DIR__.'/../db.php';
require_once __DIR__.'/../lib/auth.php';
require_auth(); if(!is_admin()) die('Only admin');

$pdo = db();

/* ------------------------- helpers ------------------------- */

function fw_safe_status_from_schema(PDO $pdo, string $table='matches'): string {
    $statusInit = 'pending';
    try {
        $col = $pdo->query("SHOW COLUMNS FROM {$table} LIKE 'status'")
                   ->fetch(PDO::FETCH_ASSOC);
        if ($col) {
            $type = strtolower($col['Type'] ?? '');
            $def  = $col['Default'] ?? null;

            if ($def !== null && $def !== '') {
                $statusInit = $def;
            }

            if (strpos($type, 'enum(') !== false) {
                preg_match_all("/'([^']+)'/", $type, $m);
                $opts  = $m[1] ?? [];
                $cands = ['pending','scheduled','open','ready','upcoming','created','init'];

                if ($def && in_array($def, $opts, true)) {
                    $statusInit = $def;
                } else {
                    foreach ($cands as $c) {
                        if (in_array($c, $opts, true)) { $statusInit = $c; break; }
                    }
                }
                if ($opts && !in_array($statusInit, $opts, true)) {
                    $statusInit = $opts[0];
                }
            }
        }
    } catch (\Throwable $e) {
        // silent fallback
    }
    return $statusInit;
}


/**
 * برنده‌ی مچ $mid را به مچ مقصد راند بعدی می‌نشاند (Single Elim).
 * پشتیبان هر دو نام ستون «اسلات»: slot یا bracket_slot
 */
function advance_winner_to_next_round(PDO $pdo, int $tid, int $mid): void {
  // کشف نام ستون‌ها
  $cols = $pdo->query("SHOW COLUMNS FROM matches")->fetchAll(PDO::FETCH_COLUMN, 0);
  $colRound = in_array('round',$cols,true) ? 'round' : null;
  $colSlot  = in_array('slot',$cols,true) ? 'slot' : (in_array('bracket_slot',$cols,true) ? 'bracket_slot' : null);
  if(!$colRound || !$colSlot) return; // بدون round/slot کاری نکن

  // مچ فعلی
  $sql = "SELECT id, {$colRound} AS r, {$colSlot} AS s, team_a_id, team_b_id, score_a, score_b
          FROM matches WHERE id=? AND tournament_id=? LIMIT 1";
  $st = $pdo->prepare($sql); $st->execute([$mid,$tid]);
  $cur = $st->fetch(PDO::FETCH_ASSOC);
  if(!$cur) return;

  $round = (int)($cur['r'] ?? 0);
  $slot  = (int)($cur['s'] ?? 0);
  if($round<=0 || $slot<=0) return;

  // محاسبه برنده (BYE ⇒ تیم A)
  $winnerId = null;
  if (!empty($cur['team_a_id']) && empty($cur['team_b_id'])) {
    $winnerId = (int)$cur['team_a_id']; // BYE
  } elseif((int)$cur['score_a'] > (int)$cur['score_b']) {
    $winnerId = (int)$cur['team_a_id'];
  } elseif((int)$cur['score_b'] > (int)$cur['score_a']) {
    $winnerId = (int)$cur['team_b_id'];
  } else {
    return; // مساوی/نامشخص
  }
  if(!$winnerId) return;

  $nextRound = $round + 1;
  $nextSlot  = intdiv($slot + 1, 2); // 1,2->1 | 3,4->2 | ...

  // پیدا/ساخت مچ مقصد
  $sel = $pdo->prepare("SELECT id, team_a_id, team_b_id FROM matches
                        WHERE tournament_id=? AND {$colRound}=? AND {$colSlot}=? LIMIT 1");
  $sel->execute([$tid,$nextRound,$nextSlot]);
  $nx = $sel->fetch(PDO::FETCH_ASSOC);

  if(!$nx){
    $statusInit = fw_safe_status_from_schema($pdo,'matches');
    $ins = $pdo->prepare("INSERT INTO matches (tournament_id, {$colRound}, {$colSlot}, team_a_id, team_b_id, status)
                          VALUES (?, ?, ?, NULL, NULL, ?)");
    $ins->execute([$tid,$nextRound,$nextSlot,$statusInit]);
    $nx = ['id'=>(int)$pdo->lastInsertId(),'team_a_id'=>null,'team_b_id'=>null];
  }

  // سمت مناسب: slot فرد ⇒ A ، زوج ⇒ B
  $putCol = ($slot % 2 === 1) ? 'team_a_id' : 'team_b_id';
  $already = array_key_exists($putCol,$nx) ? $nx[$putCol] : null;

  // فقط اگر خالی است پر کن (جلوگیری از overwrite)
  if (is_null($already) || $already === '' || $already === 0) {
    $pdo->prepare("UPDATE matches SET {$putCol}=? WHERE id=?")->execute([$winnerId, (int)$nx['id']]);
  }
}

/* ------------------------- process ------------------------- */

$matchId = (int)($_POST['id'] ?? 0);
$res     = isset($_POST['result']) && in_array($_POST['result'],['A','B'], true) ? $_POST['result'] : null;
$sa      = ($_POST['score_a'] ?? '') !== '' ? (int)$_POST['score_a'] : null;
$sb      = ($_POST['score_b'] ?? '') !== '' ? (int)$_POST['score_b'] : null;
$knife   = ($_POST['knife_winner'] ?? '') === 'B' ? 'B' : (($_POST['knife_winner'] ?? '') === 'A' ? 'A' : null);
$start   = trim($_POST['start_at'] ?? '');
$serv    = ($_POST['server_id'] ?? '') !== '' ? (int)$_POST['server_id'] : null;
$stream  = trim($_POST['stream_url'] ?? '');

$st = $pdo->prepare("SELECT * FROM matches WHERE id=?");
$st->execute([$matchId]);
$m = $st->fetch();
if(!$m) die('match not found');

try{
  $pdo->beginTransaction();

  // به‌روزرسانی فیلدهای اختیاری
  $fields=[]; $params=[];
  if($knife !== null){ $fields[]="knife_winner=?"; $params[]=$knife; }
  if($start !== '') { $fields[]="start_at=?";     $params[]=$start; }
  if($serv !== null){ $fields[]="server_id=?";    $params[]=$serv;  }
  if($stream !== ''){ $fields[]="stream_url=?";   $params[]=$stream; }
  if($fields){
    $params[]=$matchId;
    $pdo->prepare("UPDATE matches SET ".implode(',', $fields)." WHERE id=?")->execute($params);
  }

  // نتیجه
  if($res){
    if($sa === null) $sa = ($res==='A') ? 2 : 0;
    if($sb === null) $sb = ($res==='B') ? 2 : 0;

    $pdo->prepare("UPDATE matches SET status='done', score_a=?, score_b=?, finished_at=NOW() WHERE id=?")
        ->execute([$sa,$sb,$matchId]);
    audit('match.result','match',$matchId, ['old'=>$m], ['result'=>$res,'score_a'=>$sa,'score_b'=>$sb]);

    // برنده را جلو ببر
    advance_winner_to_next_round($pdo, (int)$m['tournament_id'], (int)$m['id']);

    // ---- Elo تیمی
    $A = (int)$m['team_a_id']; $B = (int)$m['team_b_id'];
    if($A && $B){
      $ua = (int)$pdo->query("SELECT elo FROM teams WHERE id=$A")->fetchColumn();
      $ub = (int)$pdo->query("SELECT elo FROM teams WHERE id=$B")->fetchColumn();
      $ua_cal = (int)$pdo->query("SELECT calibration_left FROM teams WHERE id=$A")->fetchColumn();
      $ub_cal = (int)$pdo->query("SELECT calibration_left FROM teams WHERE id=$B")->fetchColumn();
      $Kbase = 24; $Ka = $ua_cal>0?36:$Kbase; $Kb = $ub_cal>0?36:$Kbase;

      $elo = function($ra,$rb,$score,$K){ $Ea = 1/(1+pow(10,(($rb-$ra)/400))); return (int)round($ra + $K*($score-$Ea)); };
      if($res==='A'){ $na=$elo($ua,$ub,1,$Ka); $nb=$elo($ub,$ua,0,$Kb); }
      else          { $na=$elo($ua,$ub,0,$Ka); $nb=$elo($ub,$ua,1,$Kb); }

      $pdo->prepare("UPDATE teams SET elo=?, calibration_left=GREATEST(0,calibration_left-1) WHERE id=?")->execute([$na,$A]);
      $pdo->prepare("UPDATE teams SET elo=?, calibration_left=GREATEST(0,calibration_left-1) WHERE id=?")->execute([$nb,$B]);
    }

    // ---- Elo فصلی (اگر فصل فعال دارید)
    $sid = (int)$pdo->query("SELECT id FROM seasons WHERE is_active=1")->fetchColumn();
    if($sid && $A && $B){
      $pdo->prepare("INSERT IGNORE INTO season_ratings(season_id,team_id,elo,calibration_left) VALUES(?,?,1000,10)")->execute([$sid,$A]);
      $pdo->prepare("INSERT IGNORE INTO season_ratings(season_id,team_id,elo,calibration_left) VALUES(?,?,1000,10)")->execute([$sid,$B]);

      $ra = (int)$pdo->query("SELECT elo FROM season_ratings WHERE season_id=$sid AND team_id=$A")->fetchColumn();
      $rb = (int)$pdo->query("SELECT elo FROM season_ratings WHERE season_id=$sid AND team_id=$B")->fetchColumn();
      $ca = (int)$pdo->query("SELECT calibration_left FROM season_ratings WHERE season_id=$sid AND team_id=$A")->fetchColumn();
      $cb = (int)$pdo->query("SELECT calibration_left FROM season_ratings WHERE season_id=$sid AND team_id=$B")->fetchColumn();

      $Ka = $ca>0?36:24; $Kb = $cb>0?36:24;
      $elo = function($r1,$r2,$score,$K){ $E=1/(1+pow(10,(($r2-$r1)/400))); return (int)round($r1 + $K*($score-$E)); };
      if($res==='A'){ $nra=$elo($ra,$rb,1,$Ka); $nrb=$elo($rb,$ra,0,$Kb); }
      else          { $nra=$elo($ra,$rb,0,$Ka); $nrb=$elo($rb,$ra,1,$Kb); }

      $pdo->prepare("UPDATE season_ratings SET elo=?, calibration_left=GREATEST(0,calibration_left-1) WHERE season_id=? AND team_id=?")->execute([$nra,$sid,$A]);
      $pdo->prepare("UPDATE season_ratings SET elo=?, calibration_left=GREATEST(0,calibration_left-1) WHERE season_id=? AND team_id=?")->execute([$nrb,$sid,$B]);
    }

    // ---- Settlement prediction ها (pari-mutuel)
    $pool = $pdo->prepare("SELECT side, SUM(amount) s FROM predictions WHERE match_id=? AND status='open' GROUP BY side");
    $pool->execute([$matchId]);
    $sumA=$sumB=0;
    foreach($pool as $row){ if($row['side']==='A') $sumA=(int)$row['s']; else $sumB=(int)$row['s']; }
    $total = $sumA + $sumB;
    $winSide = $res; $winnersSum = ($winSide==='A') ? $sumA : $sumB;

    if($total>0){
      if($winnersSum>0){
        $stmt = $pdo->prepare("SELECT * FROM predictions WHERE match_id=? AND status='open' AND side=?");
        $stmt->execute([$matchId,$winSide]);
        foreach($stmt as $p){
          $share  = $p['amount'] * ($total / $winnersSum);
          $payout = (int)floor($share);
          $pdo->prepare("INSERT INTO wallets(owner_type,owner_id,balance)
               SELECT 'user',?,0 WHERE NOT EXISTS (SELECT 1 FROM wallets WHERE owner_type='user' AND owner_id=?)")
               ->execute([$p['user_id'],$p['user_id']]);
          $pdo->prepare("UPDATE wallets SET balance=balance+? WHERE owner_type='user' AND owner_id=?")->execute([$payout,$p['user_id']]);
          $pdo->prepare("INSERT INTO transactions(wallet_owner_type,wallet_owner_id,direction,amount,ref_type,ref_id,memo)
               VALUES('user',?,'credit',?,'prediction',?,'bet win')")->execute([$p['user_id'],$payout,$matchId]);
          $pdo->prepare("UPDATE predictions SET status='settled', payout=? WHERE id=?")->execute([$payout,$p['id']]);
        }
      } else {
        $stmt = $pdo->prepare("SELECT * FROM predictions WHERE match_id=? AND status='open'");
        $stmt->execute([$matchId]);
        foreach($stmt as $p){
          $pdo->prepare("UPDATE wallets SET balance=balance+? WHERE owner_type='user' AND owner_id=?")->execute([$p['amount'],$p['user_id']]);
          $pdo->prepare("INSERT INTO transactions(wallet_owner_type,wallet_owner_id,direction,amount,ref_type,ref_id,memo)
               VALUES('user',?,'credit',?,'prediction',?,'bet void')")->execute([$p['user_id'],$p['amount'],$matchId]);
          $pdo->prepare("UPDATE predictions SET status='void' WHERE id=?")->execute([$p['id']]);
        }
      }
    } else {
      $pdo->prepare("UPDATE predictions SET status='void' WHERE match_id=? AND status='open'")->execute([$matchId]);
    }

    // ---- سازگاری با bets (اگر وجود دارد)
    $hasBetsTbl = (bool)$pdo->query("SHOW TABLES LIKE 'bets'")->fetch();
    if($hasBetsTbl){
      $pdo->prepare("UPDATE bets SET status='lost' WHERE match_id=? AND status='pending' AND team_pick<>?")->execute([$matchId,$res]);
      $stmt=$pdo->prepare("SELECT user_id, amount FROM bets WHERE match_id=? AND status='pending' AND team_pick=? FOR UPDATE");
      $stmt->execute([$matchId,$res]);
      foreach($stmt as $b){
        $pdo->prepare("UPDATE wallets SET balance=balance+? WHERE owner_type='user' AND owner_id=?")
            ->execute([(int)$b['amount']*2,(int)$b['user_id']]);
      }
      $pdo->prepare("UPDATE bets SET status='won' WHERE match_id=? AND team_pick=? AND status='pending'")->execute([$matchId,$res]);
    }
  }

  $pdo->commit();
}catch(Throwable $e){
  if($pdo->inTransaction()) $pdo->rollBack();
  echo '<div style="padding:16px;background:#1b2330;color:#f8d7da;border:1px solid #842029;border-radius:8px;font-family:system-ui">';
  echo 'خطا در ثبت نتیجه: '.h($e->getMessage());
  echo '</div>';
  exit;
}

redirect('pages/match_view.php?id='.$matchId);
